<?php
// index.php
require_once 'config.php';

// Process login form
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['name'])) {
    $name = trim($_POST['name']);
    
    if (!empty($name)) {
        $user = saveUser($name);
        $_SESSION['user'] = $user;
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
}

// Process message sending
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['message']) && isLoggedIn()) {
    $message = trim($_POST['message']);
    
    if (!empty($message)) {
        saveMessage($_SESSION['user']['id'], $message);
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
}

// If user was already logged in, update their info
if (isLoggedIn()) {
    $currentUser = $_SESSION['user'];
    $users = getUsers();
    
    foreach ($users as &$user) {
        if ($user['id'] === $currentUser['id']) {
            $user['last_seen'] = time();
            $_SESSION['user'] = $user;
            break;
        }
    }
    
    saveUsers($users);
}

$currentUser = $_SESSION['user'] ?? null;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Support Chat System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #0084ff;
            --secondary-color: #f0f2f5;
            --text-color: #050505;
            --light-text: #65676b;
            --border-color: #e4e6eb;
            --card-bg: #ffffff;
            --hover-bg: #f2f2f2;
            --success: #00b900;
            --radius: 18px;
            --shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Roboto, Helvetica, Arial, sans-serif;
        }

        body {
            background-color: #f0f2f5;
            color: var(--text-color);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }

        .container {
            width: 100%;
            max-width: 1000px;
            height: 90vh;
            display: flex;
            box-shadow: var(--shadow);
            border-radius: var(--radius);
            overflow: hidden;
        }

        .sidebar {
            width: 35%;
            background-color: var(--card-bg);
            border-right: 1px solid var(--border-color);
            display: flex;
            flex-direction: column;
        }

        .main-content {
            width: 65%;
            background-color: var(--card-bg);
            display: flex;
            flex-direction: column;
        }

        .header {
            padding: 16px 20px;
            background-color: var(--card-bg);
            border-bottom: 1px solid var(--border-color);
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-weight: 700;
            font-size: 20px;
            color: var(--primary-color);
        }

        .user-info {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: var(--primary-color);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 18px;
        }

        .avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }

        .search-container {
            padding: 12px;
            border-bottom: 1px solid var(--border-color);
        }

        .search-box {
            display: flex;
            align-items: center;
            background-color: var(--secondary-color);
            border-radius: 20px;
            padding: 8px 16px;
        }

        .search-box input {
            flex: 1;
            border: none;
            background: transparent;
            padding: 8px;
            outline: none;
            font-size: 14px;
        }

        .chat-list {
            flex: 1;
            overflow-y: auto;
        }

        .chat-item {
            display: flex;
            align-items: center;
            padding: 12px 16px;
            cursor: pointer;
            transition: background-color 0.2s;
        }

        .chat-item:hover {
            background-color: var(--hover-bg);
        }

        .chat-item.active {
            background-color: var(--secondary-color);
        }

        .chat-info {
            flex: 1;
            margin-left: 12px;
        }

        .chat-name {
            font-weight: 600;
            font-size: 15px;
        }

        .chat-preview {
            font-size: 13px;
            color: var(--light-text);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 180px;
        }

        .chat-time {
            font-size: 12px;
            color: var(--light-text);
        }

        .chat-area {
            flex: 1;
            padding: 20px;
            overflow-y: auto;
            display: flex;
            flex-direction: column;
            gap: 12px;
            background-color: var(--secondary-color);
        }

        .message {
            max-width: 70%;
            padding: 10px 16px;
            border-radius: 18px;
            position: relative;
            animation: fadeIn 0.3s ease;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .message-in {
            align-self: flex-start;
            background-color: var(--card-bg);
            border-bottom-left-radius: 4px;
        }

        .message-out {
            align-self: flex-end;
            background-color: var(--primary-color);
            color: white;
            border-bottom-right-radius: 4px;
        }

        .message-time {
            font-size: 11px;
            margin-top: 4px;
            text-align: right;
            opacity: 0.8;
        }

        .input-area {
            padding: 16px;
            background-color: var(--card-bg);
            border-top: 1px solid var(--border-color);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .message-input {
            flex: 1;
            padding: 12px 18px;
            border: none;
            background-color: var(--secondary-color);
            border-radius: 20px;
            outline: none;
            font-size: 15px;
        }

        .send-button {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: var(--primary-color);
            color: white;
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
        }

        .login-container {
            width: 100%;
            max-width: 400px;
            background-color: var(--card-bg);
            border-radius: var(--radius);
            padding: 30px;
            box-shadow: var(--shadow);
        }

        .login-title {
            text-align: center;
            margin-bottom: 24px;
            color: var(--primary-color);
            font-size: 24px;
        }

        .login-form {
            display: flex;
            flex-direction: column;
            gap: 16px;
        }

        .form-group {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }

        .form-group label {
            font-size: 14px;
            color: var(--light-text);
        }

        .form-control {
            padding: 14px 16px;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            font-size: 16px;
            outline: none;
            transition: border-color 0.2s;
        }

        .form-control:focus {
            border-color: var(--primary-color);
        }

        .btn {
            padding: 14px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.2s;
        }

        .btn-primary {
            background-color: var(--primary-color);
            color: white;
        }

        .btn-primary:hover {
            background-color: #0066cc;
        }

        .welcome-text {
            text-align: center;
            margin-top: 20px;
            color: var(--light-text);
            font-size: 14px;
        }

        .status {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 13px;
        }

        .status-indicator {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background-color: var(--success);
        }

        .hidden {
            display: none;
        }

        .admin-link {
            display: block;
            text-align: center;
            margin-top: 15px;
            color: var(--primary-color);
            text-decoration: none;
        }

        @media (max-width: 768px) {
            .container {
                flex-direction: column;
                height: auto;
            }
            
            .sidebar, .main-content {
                width: 100%;
            }
            
            .sidebar {
                height: 40vh;
            }
            
            .main-content {
                height: 60vh;
            }
        }
    </style>
</head>
<body>
    <?php if (!$currentUser): ?>
        <div class="login-container">
            <h1 class="login-title"><i class="fas fa-headset"></i> Support Chat</h1>
            <div class="login-form">
                <div class="form-group">
                    <label for="username">Your Name</label>
                    <input type="text" id="username" name="name" class="form-control" placeholder="Enter your name" required>
                </div>
                <button class="btn btn-primary" id="loginBtn">
                    Start Chat
                </button>
            </div>
            <p class="welcome-text">Welcome to our support system. We're here to help you!</p>
                    </div>
    <?php else: ?>
        <div class="container">
            <div class="sidebar">
                <div class="header">
                    <div class="logo">
                        <i class="fas fa-headset"></i>
                        <span>Support</span>
                    </div>
                    <div class="user-info">
                        <div class="avatar"><?= substr($currentUser['name'], 0, 1) ?></div>
                        <span><?= $currentUser['name'] ?></span>
                    </div>
                </div>
                
                <div class="search-container">
                    <div class="search-box">
                        <i class="fas fa-search"></i>
                        <input type="text" placeholder="Search conversations...">
                    </div>
                </div>
                
                <div class="chat-list">
                    <div class="chat-item active">
                        <div class="avatar" style="background-color: #0084ff;">S</div>
                        <div class="chat-info">
                            <div class="chat-name">Support Team</div>
                            <div class="chat-preview">How can I help you today?</div>
                        </div>
                        <div class="chat-time">Now</div>
                    </div>
                </div>
            </div>
            
            <div class="main-content">
                <div class="header">
                    <div class="user-info">
                        <div class="avatar" style="background-color: #0084ff;">S</div>
                        <div>
                            <div class="chat-name">Support Team</div>
                            <div class="status">
                                <div class="status-indicator"></div>
                                <span>Online</span>
                            </div>
                        </div>
                    </div>
                    <div>
                        <i class="fas fa-phone-alt" style="margin-right: 16px; cursor: pointer;"></i>
                        <i class="fas fa-video" style="cursor: pointer;"></i>
                    </div>
                </div>
                
                <div class="chat-area" id="chatArea">
                    <?php
                    $messages = getUserMessages($currentUser['id']);
                    foreach ($messages as $message):
                    ?>
                        <div class="message <?= $message['is_admin'] ? 'message-in' : 'message-out' ?>">
                            <div><?= htmlspecialchars($message['message']) ?></div>
                            <div class="message-time"><?= date('H:i', $message['timestamp']) ?></div>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <form method="POST" class="input-area">
                    <i class="far fa-smile" style="cursor: pointer;"></i>
                    <i class="fas fa-paperclip" style="cursor: pointer; margin-left: 12px;"></i>
                    <input type="text" name="message" class="message-input" placeholder="Type a message...">
                    <button type="submit" class="send-button">
                        <i class="fas fa-paper-plane"></i>
                    </button>
                </form>
            </div>
        </div>
    <?php endif; ?>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const loginBtn = document.getElementById('loginBtn');
            
            if (loginBtn) {
                loginBtn.addEventListener('click', function() {
                    const username = document.getElementById('username').value.trim();
                    if (username) {
                        // Create form and submit it
                        const form = document.createElement('form');
                        form.method = 'POST';
                        form.action = '';
                        
                        const input = document.createElement('input');
                        input.type = 'hidden';
                        input.name = 'name';
                        input.value = username;
                        
                        form.appendChild(input);
                        document.body.appendChild(form);
                        form.submit();
                    }
                });
            }
            
            // Scroll to bottom of chat
            const chatArea = document.getElementById('chatArea');
            if (chatArea) {
                chatArea.scrollTop = chatArea.scrollHeight;
            }
        });
    </script>
</body>
</html>